/* global chrome */

const STORAGE_KEYS = {
  RULES: 'rulesConfig',
  SETTINGS: 'userSettings',
  GLOBAL: 'globalState'
}

const RUNTIME_PREFIX = 'runtime:'

const DEFAULT_SETTINGS = Object.freeze({
  defaultIntervalMs: 60000,
  defaultJitter: 0.03,
  defaultStopAfter: null,
  guards: {
    pauseOnTyping: true,
    pauseOnMedia: true,
    pauseOnUpload: true
  },
  quietHours: {
    enabled: false,
    start: '22:00',
    end: '07:00'
  },
  keepActiveDefault: false,
  maxJitter: 0.05,
  minIntervalMs: 5000
})

const DEFAULT_GLOBAL_STATE = Object.freeze({
  paused: false,
  pausedAt: null,
  quickResetAt: null
})

const ALARM_PREFIX = 'rule:'
const CHECK_ALARM_PREFIX = 'check:'
const BADGE_TICK_MS = 1000
const COMMAND_INTERVAL_STEP_MS = 5000
const BADGE_INACTIVE_TEXT = ''
const ICON_SPINNER_INTERVAL_MS = 180
const ICON_DEFAULT_PATHS = {
  16: 'img/icons/16.png',
  32: 'img/icons/32.png',
  48: 'img/icons/48.png',
  128: 'img/icons/128.png'
}
const ICON_ACTIVE_PATHS = {
  16: 'img/icons/active-16.png',
  32: 'img/icons/active-32.png',
  48: 'img/icons/active-48.png',
  128: 'img/icons/active-128.png'
}
const ICON_SPINNER_PATHS = [
  {
    16: 'img/icons/spinner-0-16.png',
    32: 'img/icons/spinner-0-32.png',
    48: 'img/icons/spinner-0-48.png',
    128: 'img/icons/spinner-0-128.png'
  },
  {
    16: 'img/icons/spinner-1-16.png',
    32: 'img/icons/spinner-1-32.png',
    48: 'img/icons/spinner-1-48.png',
    128: 'img/icons/spinner-1-128.png'
  },
  {
    16: 'img/icons/spinner-2-16.png',
    32: 'img/icons/spinner-2-32.png',
    48: 'img/icons/spinner-2-48.png',
    128: 'img/icons/spinner-2-128.png'
  },
  {
    16: 'img/icons/spinner-3-16.png',
    32: 'img/icons/spinner-3-32.png',
    48: 'img/icons/spinner-3-48.png',
    128: 'img/icons/spinner-3-128.png'
  }
]
const CHANGE_LOG_LIMIT = 50

const state = {
  initialized: false,
  rules: new Map(), // tabId (number) => config object persisted in sync
  runtime: new Map(), // tabId (number) => runtime state persisted in local
  settings: { ...DEFAULT_SETTINGS },
  global: { ...DEFAULT_GLOBAL_STATE },
  badgeIntervalId: null,
  testMode: false,
  iconAnimations: new Map()
}

chrome.runtime.onInstalled.addListener(async ({ reason }) => {
  if (reason === 'install') {
    await chrome.storage.sync.set({
      [STORAGE_KEYS.SETTINGS]: DEFAULT_SETTINGS,
      [STORAGE_KEYS.GLOBAL]: DEFAULT_GLOBAL_STATE
    })
  }

  await chrome.action.setBadgeBackgroundColor({ color: '#1d4ed8' })
  await chrome.action.setIcon({ path: ICON_DEFAULT_PATHS })
  await bootstrap()
})

// Ensure state is loaded even if service worker started outside install event.
bootstrap().catch(err => {
  console.error('Ultra Reload bootstrap failed', err)
})

async function bootstrap () {
  if (state.initialized) return

  const { [STORAGE_KEYS.RULES]: storedRules = {}, [STORAGE_KEYS.SETTINGS]: storedSettings, [STORAGE_KEYS.GLOBAL]: storedGlobal } = await chrome.storage.sync.get([STORAGE_KEYS.RULES, STORAGE_KEYS.SETTINGS, STORAGE_KEYS.GLOBAL])

  state.settings = {
    ...DEFAULT_SETTINGS,
    ...(storedSettings || {})
  }

  state.global = {
    ...DEFAULT_GLOBAL_STATE,
    ...(storedGlobal || {})
  }

  const tabIds = []
  for (const [tabIdStr, config] of Object.entries(storedRules)) {
    const tabId = Number(tabIdStr)
    if (Number.isNaN(tabId)) continue
    state.rules.set(tabId, normalizeRuleConfig(tabId, config))
    tabIds.push(tabId)
  }

  if (tabIds.length > 0) {
    const runtimeEntries = await chrome.storage.local.get(tabIds.map(tabId => `${RUNTIME_PREFIX}${tabId}`))
    for (const tabId of tabIds) {
      const runtimeKey = `${RUNTIME_PREFIX}${tabId}`
      const savedRuntime = runtimeEntries[runtimeKey]
      state.runtime.set(tabId, createRuntimeState(tabId, savedRuntime))
      if (state.rules.get(tabId)?.active) {
        scheduleRule(tabId, 'resume')
      }
    }
    ensureBadgeUpdater()
  }

  state.initialized = true
}

async function ensureInitialized () {
  if (state.initialized) return
  await bootstrap()
}

function setActionIcon (tabId, type, frameIndex = 0) {
  let path
  switch (type) {
    case 'active':
      path = ICON_ACTIVE_PATHS
      break
    case 'spinner':
      path = ICON_SPINNER_PATHS[frameIndex % ICON_SPINNER_PATHS.length]
      break
    case 'default':
    default:
      path = ICON_DEFAULT_PATHS
      break
  }

  const details = { path }
  if (typeof tabId === 'number') {
    details.tabId = tabId
  }

  chrome.action.setIcon(details).catch(() => {})
}

function setDefaultIcon (tabId) {
  setActionIcon(tabId, 'default')
}

function setActiveIcon (tabId) {
  setActionIcon(tabId, 'active')
}

function stopIconSpinner (tabId) {
  const timer = state.iconAnimations.get(tabId)
  if (timer) {
    clearInterval(timer)
    state.iconAnimations.delete(tabId)
  }
}

function startIconSpinner (tabId) {
  stopIconSpinner(tabId)
  let frame = 0
  const updateFrame = () => {
    setActionIcon(tabId, 'spinner', frame)
    frame = (frame + 1) % ICON_SPINNER_PATHS.length
  }
  updateFrame()
  const timer = setInterval(updateFrame, ICON_SPINNER_INTERVAL_MS)
  state.iconAnimations.set(tabId, timer)
}

function normalizeRuleConfig (tabId, config = {}) {
  const guards = {
    ...DEFAULT_SETTINGS.guards,
    ...(config.guards || {})
  }

  return {
    tabId,
    origin: config.origin || null,
    intervalMs: clampInterval(config.intervalMs ?? DEFAULT_SETTINGS.defaultIntervalMs),
    hardReload: Boolean(config.hardReload),
    jitter: clampJitter(config.jitter ?? state.settings.defaultJitter ?? DEFAULT_SETTINGS.defaultJitter),
    stopAfter: config.stopAfter === null || config.stopAfter === undefined ? null : Math.max(1, Number(config.stopAfter)),
    keepActive: Boolean(config.keepActive ?? state.settings.keepActiveDefault),
    guards,
    monitors: {
      text: {
        enabled: Boolean(config?.monitors?.text?.enabled),
        selector: config?.monitors?.text?.selector || '',
        match: config?.monitors?.text?.match || '',
        stopOnMatch: config?.monitors?.text?.stopOnMatch ?? true
      },
      title: {
        enabled: Boolean(config?.monitors?.title?.enabled),
        stopOnChange: config?.monitors?.title?.stopOnChange ?? true
      }
    },
    changeLog: Array.isArray(config.changeLog) ? config.changeLog.slice(-50) : [],
    active: Boolean(config.active),
    startedAt: config.startedAt || null,
    lastRunAt: config.lastRunAt || null
  }
}

function createRuntimeState (tabId, saved = {}) {
  return {
    tabId,
    status: saved.status || 'idle',
    guardState: {
      typing: false,
      media: false,
      upload: false,
      hidden: false,
      audible: false,
      ...(saved.guardState || {})
    },
    manualPause: Boolean(saved.manualPause),
    offline: Boolean(saved.offline),
    nextRunAt: saved.nextRunAt || null,
    remainingRuns: saved.remainingRuns === null || saved.remainingRuns === undefined ? null : Number(saved.remainingRuns),
    lastMessageAt: saved.lastMessageAt || null,
    wasDiscarded: Boolean(saved.wasDiscarded),
    lastDiscardedAt: saved.lastDiscardedAt || null,
    textSnapshot: saved.textSnapshot || null,
    titleBaseline: saved.titleBaseline || null,
    changeCount: saved.changeCount || 0,
    countdownAccessedAt: Date.now()
  }
}

function clampInterval (value) {
  const min = state.settings.minIntervalMs ?? DEFAULT_SETTINGS.minIntervalMs
  const max = 24 * 60 * 60 * 1000 // 24h
  const numeric = Number(value)
  if (Number.isNaN(numeric)) return min
  return Math.min(Math.max(numeric, min), max)
}

function clampJitter (value) {
  const numeric = Number(value)
  if (Number.isNaN(numeric)) return DEFAULT_SETTINGS.defaultJitter
  const max = state.settings.maxJitter ?? DEFAULT_SETTINGS.maxJitter
  return Math.min(Math.max(numeric, 0), max)
}

async function persistRules () {
  const serializable = {}
  for (const [tabId, config] of state.rules.entries()) {
    serializable[tabId] = {
      ...config,
      tabId: undefined // omit redundant field
    }
  }
  await chrome.storage.sync.set({ [STORAGE_KEYS.RULES]: serializable })
}

async function persistRuntime (tabId) {
  const runtime = state.runtime.get(tabId)
  if (!runtime) return
  const payload = {
    status: runtime.status,
    guardState: runtime.guardState,
    manualPause: runtime.manualPause,
    offline: runtime.offline,
    nextRunAt: runtime.nextRunAt,
    remainingRuns: runtime.remainingRuns,
    lastMessageAt: runtime.lastMessageAt,
    wasDiscarded: runtime.wasDiscarded,
    lastDiscardedAt: runtime.lastDiscardedAt,
    textSnapshot: runtime.textSnapshot,
    titleBaseline: runtime.titleBaseline,
    changeCount: runtime.changeCount
  }
  await chrome.storage.local.set({ [`${RUNTIME_PREFIX}${tabId}`]: payload })
}

async function clearRuntime (tabId) {
  state.runtime.delete(tabId)
  await chrome.storage.local.remove(`${RUNTIME_PREFIX}${tabId}`)
}

function isRuleGuarded (runtime, config) {
  if (!runtime) return false
  const { guardState, manualPause, offline } = runtime
  if (manualPause || offline) return true

  if (state.global.paused) return true

  if (isQuietHoursActive()) return true

  if ((config.guards.pauseOnTyping && guardState.typing) ||
    (config.guards.pauseOnMedia && (guardState.media || guardState.audible)) ||
    (config.guards.pauseOnUpload && guardState.upload)) {
    return true
  }

  return false
}

function isQuietHoursActive () {
  const quiet = state.settings.quietHours
  if (!quiet?.enabled) return false
  const now = new Date()
  const [startHour, startMinute] = quiet.start.split(':').map(Number)
  const [endHour, endMinute] = quiet.end.split(':').map(Number)
  const start = new Date(now)
  start.setHours(startHour, startMinute || 0, 0, 0)
  const end = new Date(now)
  end.setHours(endHour, endMinute || 0, 0, 0)

  if (quiet.start === quiet.end) {
    return true
  }

  if (start < end) {
    return now >= start && now < end
  }

  return now >= start || now < end
}

async function scheduleRule (tabId, reason = 'start') {
  const config = state.rules.get(tabId)
  if (!config || !config.active) return

  const runtime = state.runtime.get(tabId) || createRuntimeState(tabId)
  state.runtime.set(tabId, runtime)

  const interval = clampInterval(config.intervalMs)
  const jitterDelta = computeJitter(interval, config.jitter)
  const nextInMs = interval + jitterDelta

  const alarmName = `${ALARM_PREFIX}${tabId}`
  runtime.nextRunAt = Date.now() + nextInMs
  runtime.status = 'scheduled'
  await chrome.alarms.create(alarmName, { when: runtime.nextRunAt })

  await persistRuntime(tabId)
  ensureBadgeUpdater()
  console.debug('[UltraReload] Scheduled rule', { tabId, reason, interval, jitterDelta })
}

function computeJitter (interval, jitterRatio) {
  const ratio = clampJitter(jitterRatio)
  if (ratio <= 0) return 0
  const variance = interval * ratio
  return Math.round((Math.random() * variance * 2) - variance)
}

async function cancelRuleAlarm (tabId) {
  const alarmName = `${ALARM_PREFIX}${tabId}`
  await chrome.alarms.clear(alarmName)
}

async function handleAlarm (alarm) {
  if (!alarm?.name) return
  if (alarm.name.startsWith(ALARM_PREFIX)) {
    const tabId = Number(alarm.name.replace(ALARM_PREFIX, ''))
    if (!Number.isNaN(tabId)) {
      await executeRule(tabId)
    }
  } else if (alarm.name.startsWith(CHECK_ALARM_PREFIX)) {
    const tabId = Number(alarm.name.replace(CHECK_ALARM_PREFIX, ''))
    if (!Number.isNaN(tabId)) {
      await evaluateConditions(tabId, 'check-alarm')
    }
  }
}

chrome.alarms.onAlarm.addListener(alarm => {
  handleAlarm(alarm).catch(err => {
    console.error('Ultra Reload alarm error', alarm?.name, err)
  })
})

async function executeRule (tabId) {
  const config = state.rules.get(tabId)
  const runtime = state.runtime.get(tabId)
  if (!config || !config.active || !runtime) {
    await cancelRuleAlarm(tabId)
    return
  }

  runtime.status = 'evaluating'
  await persistRuntime(tabId)

  const allowed = await evaluateConditions(tabId, 'alarm-trigger')
  if (!allowed) {
    await chrome.alarms.create(`${CHECK_ALARM_PREFIX}${tabId}`, { when: Date.now() + 3000 })
    return
  }

  const tab = await safeGetTab(tabId)
  if (!tab) {
    await stopRule(tabId, { reason: 'tab-missing' })
    return
  }

  const monitorResult = await evaluateMonitors(tabId, config, runtime, tab)
  if (monitorResult.stop) {
    if (monitorResult.notification) {
      await sendNotification(monitorResult.notification.title, monitorResult.notification.message)
    }
    await stopRule(tabId, { reason: monitorResult.reason, badgeText: monitorResult.badgeText })
    return
  }

  runtime.status = 'reloading'
  await persistRuntime(tabId)

  try {
    await performReload(tabId, config)
    runtime.status = 'completed'
    runtime.wasDiscarded = false
    runtime.offline = false
    runtime.nextRunAt = null
    const now = Date.now()
    config.lastRunAt = now
    if (!runtime.remainingRuns && config.stopAfter) {
      runtime.remainingRuns = config.stopAfter
    }
    if (runtime.remainingRuns !== null) {
      runtime.remainingRuns = Math.max(0, runtime.remainingRuns - 1)
      if (runtime.remainingRuns === 0) {
        await stopRule(tabId, { reason: 'limit-reached' })
        return
      }
    }
    await persistRuntime(tabId)
    await persistRules()
  } catch (error) {
    console.error('Ultra Reload failed to reload tab', tabId, error)
    runtime.status = 'error'
    runtime.lastError = {
      message: error.message,
      at: Date.now()
    }
    await persistRuntime(tabId)
  } finally {
    if (state.rules.get(tabId)?.active) {
      await scheduleRule(tabId, 'post-reload')
    }
  }
}

async function evaluateConditions (tabId, _trigger) {
  const config = state.rules.get(tabId)
  const runtime = state.runtime.get(tabId)
  if (!config || !runtime) return false

  const guardBlock = isRuleGuarded(runtime, config)
  if (guardBlock) {
    runtime.status = 'guard-paused'
    await persistRuntime(tabId)
    return false
  }

  if (config.stopAfter && runtime.remainingRuns === 0) {
    return false
  }

  const tab = await safeGetTab(tabId)
  if (!tab) {
    await stopRule(tabId, { reason: 'tab-missing' })
    return false
  }

  if (!isTabUrlSupported(tab.url)) {
    await stopRule(tabId, { reason: 'unsupported' })
    return false
  }

  if (runtime.offline) {
    runtime.status = 'offline-paused'
    await persistRuntime(tabId)
    return false
  }

  if (runtime.wasDiscarded) {
    runtime.status = 'discarded'
    await persistRuntime(tabId)
    return false
  }

  runtime.status = 'ready'
  await persistRuntime(tabId)
  return true
}

async function performReload (tabId, config) {
  const updateOps = []
  if (config.keepActive) {
    updateOps.push(chrome.tabs.update(tabId, { autoDiscardable: false }).catch(() => {}))
  }

  await Promise.all(updateOps)

  const reloadOptions = {
    bypassCache: Boolean(config.hardReload)
  }

  /**
   * MDN docs confirm tabs.reload supports bypassCache for hard refresh:
   * https://developer.mozilla.org/en-US/docs/Mozilla/Add-ons/WebExtensions/API/tabs/reload
   */
  await chrome.tabs.reload(tabId, reloadOptions)
  await chrome.action.setBadgeText({ text: '⟳', tabId })
}

async function isHostPermissionGranted (origin) {
  if (!origin) return false
  const pattern = `${origin}/*`
  return chrome.permissions.contains({ origins: [pattern] })
}

async function startRule (tabId, options = {}) {
  await ensureInitialized()
  const tab = await safeGetTab(tabId)
  if (!tab || !isTabUrlSupported(tab.url)) {
    throw new Error('Current tab URL is not supported for auto refresh.')
  }

  const origin = new URL(tab.url).origin
  const hasPermission = state.testMode ? true : await isHostPermissionGranted(origin)
  if (!hasPermission) {
    const error = new Error('Site permission is required to start auto refresh. Use the popup to grant access and try again.')
    error.code = 'HOST_PERMISSION_REQUIRED'
    throw error
  }

  const config = state.rules.get(tabId) || normalizeRuleConfig(tabId, {})
  config.origin = origin
  config.intervalMs = clampInterval(options.intervalMs ?? config.intervalMs)
  config.hardReload = Boolean(options.hardReload ?? config.hardReload)
  config.jitter = clampJitter(options.jitter ?? config.jitter)
  config.stopAfter = options.stopAfter === null || options.stopAfter === undefined ? null : Math.max(1, Number(options.stopAfter))
  config.keepActive = Boolean(options.keepActive ?? config.keepActive)
  config.guards = {
    ...config.guards,
    ...(options.guards || {})
  }
  if (options.monitors) {
    config.monitors = {
      text: {
        ...config.monitors?.text,
        ...(options.monitors.text || {})
      },
      title: {
        ...config.monitors?.title,
        ...(options.monitors.title || {})
      }
    }
  }
  config.active = true
  config.startedAt = Date.now()

  const runtime = createRuntimeState(tabId, {
    remainingRuns: config.stopAfter,
    titleBaseline: null
  })
  runtime.manualPause = false
  runtime.status = 'starting'
  state.rules.set(tabId, config)
  state.runtime.set(tabId, runtime)

  await persistRules()
  await persistRuntime(tabId)

  await initializeMonitors(tabId, config, runtime, tab)
  await injectGuardScript(tabId)
  await scheduleRule(tabId, 'start')
  await chrome.action.setBadgeText({ text: '…', tabId })
  startIconSpinner(tabId)

  return { config, runtime }
}

async function injectGuardScript (tabId) {
  try {
    await chrome.scripting.executeScript({
      target: { tabId },
      files: ['content/guard.js']
    })
  } catch (error) {
    console.warn('Ultra Reload guard injection failed', tabId, error)
  }
}

async function stopRule (tabId, { reason = 'user', badgeText } = {}) {
  await ensureInitialized()
  const config = state.rules.get(tabId)
  if (!config) return
  config.active = false
  config.startedAt = config.startedAt || null
  stopIconSpinner(tabId)
  setDefaultIcon(tabId)
  await cancelRuleAlarm(tabId)
  await chrome.alarms.clear(`${CHECK_ALARM_PREFIX}${tabId}`)
  await persistRules()
  await clearRuntime(tabId)
  const text = badgeText !== undefined ? String(badgeText) : BADGE_INACTIVE_TEXT
  await chrome.action.setBadgeText({ text, tabId })
  console.debug('[UltraReload] Stopped rule', { tabId, reason })
  if (!Array.from(state.rules.values()).some(rule => rule.active)) {
    if (state.badgeIntervalId) {
      clearInterval(state.badgeIntervalId)
      state.badgeIntervalId = null
    }
  }
}

async function pauseRule (tabId) {
  await ensureInitialized()
  const runtime = state.runtime.get(tabId)
  if (!runtime) return
  runtime.manualPause = true
  runtime.status = 'manual-pause'
  await cancelRuleAlarm(tabId)
  await persistRuntime(tabId)
  await chrome.action.setBadgeText({ text: 'II', tabId })
  stopIconSpinner(tabId)
  setActiveIcon(tabId)
}

async function resumeRule (tabId) {
  await ensureInitialized()
  const config = state.rules.get(tabId)
  const runtime = state.runtime.get(tabId)
  if (!config || !runtime) return
  runtime.manualPause = false
  runtime.status = 'resuming'
  await persistRuntime(tabId)
  await scheduleRule(tabId, 'resume-manual')
  startIconSpinner(tabId)
}

async function pauseAll () {
  await ensureInitialized()
  state.global.paused = true
  state.global.pausedAt = Date.now()
  await chrome.storage.sync.set({ [STORAGE_KEYS.GLOBAL]: state.global })
  for (const tabId of state.rules.keys()) {
    await cancelRuleAlarm(tabId)
    const runtime = state.runtime.get(tabId)
    if (runtime) {
      runtime.status = 'global-pause'
      await persistRuntime(tabId)
      await chrome.action.setBadgeText({ text: '||', tabId })
      stopIconSpinner(tabId)
      setActiveIcon(tabId)
    }
  }
}

async function resumeAll () {
  await ensureInitialized()
  state.global.paused = false
  state.global.pausedAt = null
  await chrome.storage.sync.set({ [STORAGE_KEYS.GLOBAL]: state.global })
  for (const tabId of state.rules.keys()) {
    const config = state.rules.get(tabId)
    if (config?.active) {
      await scheduleRule(tabId, 'global-resume')
      startIconSpinner(tabId)
    }
  }
  ensureBadgeUpdater()
}

async function resetAll () {
  await ensureInitialized()
  for (const tabId of Array.from(state.rules.keys())) {
    await stopRule(tabId, { reason: 'global-reset' })
  }
  state.rules.clear()
  state.runtime.clear()
  if (state.badgeIntervalId) {
    clearInterval(state.badgeIntervalId)
    state.badgeIntervalId = null
  }
  await chrome.storage.sync.set({
    [STORAGE_KEYS.RULES]: {},
    [STORAGE_KEYS.GLOBAL]: state.global
  })
}

function ensureBadgeUpdater () {
  if (state.badgeIntervalId) return
  if (state.rules.size === 0) return

  state.badgeIntervalId = setInterval(async () => {
    if (state.rules.size === 0) {
      clearInterval(state.badgeIntervalId)
      state.badgeIntervalId = null
      return
    }

    for (const [tabId, config] of state.rules.entries()) {
      if (!config.active) continue
      const runtime = state.runtime.get(tabId)
      if (!runtime?.nextRunAt) continue
      const remainingMs = runtime.nextRunAt - Date.now()
      const text = formatBadgeText(remainingMs)
      await chrome.action.setBadgeText({ text, tabId })
    }
  }, BADGE_TICK_MS)
}

function formatBadgeText (remainingMs) {
  const remainingSec = Math.max(0, Math.floor(remainingMs / 1000))
  if (remainingSec < 60) {
    return remainingSec.toString().padStart(2, '0')
  }
  const minutes = Math.floor(remainingSec / 60)
  const seconds = remainingSec % 60
  if (minutes < 10) {
    return `${minutes}:${seconds.toString().padStart(2, '0')}`
  }
  return '99'
}

async function safeGetTab (tabId) {
  try {
    return await chrome.tabs.get(tabId)
  } catch {
    return null
  }
}

function isTabUrlSupported (url) {
  if (!url) return false
  try {
    const parsed = new URL(url)
    return ['http:', 'https:'].includes(parsed.protocol)
  } catch {
    return false
  }
}

chrome.tabs.onRemoved.addListener(async tabId => {
  stopIconSpinner(tabId)
  setDefaultIcon(tabId)
  if (!state.rules.has(tabId)) return
  await stopRule(tabId, { reason: 'tab-closed' })
})

chrome.tabs.onActivated.addListener(async activeInfo => {
  const runtime = state.runtime.get(activeInfo.tabId)
  if (runtime?.wasDiscarded) {
    runtime.wasDiscarded = false
    runtime.status = 'active'
    await persistRuntime(activeInfo.tabId)
    await scheduleRule(activeInfo.tabId, 'tab-activated')
  }
})

chrome.tabs.onUpdated.addListener(async (tabId, changeInfo, _tab) => {
  if (!state.rules.has(tabId)) return

  if (changeInfo.status === 'complete') {
    await injectGuardScript(tabId)
  }

  if (typeof changeInfo.audible === 'boolean') {
    const runtime = state.runtime.get(tabId)
    if (runtime) {
      runtime.guardState.audible = changeInfo.audible
      await persistRuntime(tabId)
    }
  }

  if (typeof changeInfo.discarded === 'boolean') {
    const runtime = state.runtime.get(tabId) || createRuntimeState(tabId)
    runtime.wasDiscarded = changeInfo.discarded
    runtime.lastDiscardedAt = changeInfo.discarded ? Date.now() : runtime.lastDiscardedAt
    state.runtime.set(tabId, runtime)
    await persistRuntime(tabId)
    if (!changeInfo.discarded) {
      await scheduleRule(tabId, 'restore-from-discard')
    }
  }
})

chrome.commands.onCommand.addListener(async command => {
  const [currentTab] = await chrome.tabs.query({ active: true, currentWindow: true })
  if (!currentTab) return
  const tabId = currentTab.id
  if (!tabId) return

  const config = state.rules.get(tabId)

  switch (command) {
    case 'toggle-timer':
      if (config?.active) {
        await stopRule(tabId, { reason: 'command-toggle-stop' })
      } else {
        await startRule(tabId, {})
      }
      break
    case 'increase-interval':
      if (!config) return
      config.intervalMs = clampInterval(config.intervalMs + COMMAND_INTERVAL_STEP_MS)
      await persistRules()
      await scheduleRule(tabId, 'command-increase')
      break
    case 'decrease-interval':
      if (!config) return
      config.intervalMs = clampInterval(config.intervalMs - COMMAND_INTERVAL_STEP_MS)
      await persistRules()
      await scheduleRule(tabId, 'command-decrease')
      break
    default:
      break
  }
})

chrome.runtime.onMessage.addListener((message, sender, sendResponse) => {
  const handler = messageHandlers[message?.type]
  if (!handler) return false
  handler(message, sender).then(response => {
    sendResponse(response)
  }).catch(err => {
    console.error('Ultra Reload message error', message?.type, err)
    sendResponse({ ok: false, error: err.message, code: err.code })
  })
  return true
})

const messageHandlers = {
  async GET_STATE (message, sender) {
    await ensureInitialized()
    const tabId = message.tabId ?? sender.tab?.id
    if (!tabId) {
      return { ok: false, error: 'Missing tabId for guard update' }
    }
    const config = tabId ? state.rules.get(tabId) : null
    const runtime = tabId ? state.runtime.get(tabId) : null
    return {
      ok: true,
      data: {
        tabId,
        config,
        runtime,
        settings: state.settings,
        global: state.global
      }
    }
  },
  async START_RULE (message) {
    await ensureInitialized()
    const { tabId, options } = message
    const result = await startRule(tabId, options)
    return { ok: true, data: result }
  },
  async STOP_RULE (message) {
    await ensureInitialized()
    await stopRule(message.tabId, { reason: message.reason || 'user' })
    return { ok: true }
  },
  async PAUSE_RULE (message) {
    await ensureInitialized()
    await pauseRule(message.tabId)
    return { ok: true }
  },
  async RESUME_RULE (message) {
    await ensureInitialized()
    await resumeRule(message.tabId)
    return { ok: true }
  },
  async UPDATE_RULE (message) {
    await ensureInitialized()
    const config = state.rules.get(message.tabId)
    if (!config) return { ok: false, error: 'Rule not found' }
    Object.assign(config, {
      intervalMs: message.options.intervalMs !== undefined ? clampInterval(message.options.intervalMs) : config.intervalMs,
      hardReload: message.options.hardReload ?? config.hardReload,
      jitter: message.options.jitter !== undefined ? clampJitter(message.options.jitter) : config.jitter,
      stopAfter: message.options.stopAfter === null || message.options.stopAfter === undefined ? null : Math.max(1, Number(message.options.stopAfter)),
      keepActive: message.options.keepActive ?? config.keepActive
    })
    if (message.options.guards) {
      config.guards = {
        ...config.guards,
        ...message.options.guards
      }
    }
    if (message.options.monitors) {
      config.monitors = {
        text: {
          ...config.monitors?.text,
          ...(message.options.monitors.text || {})
        },
        title: {
          ...config.monitors?.title,
          ...(message.options.monitors.title || {})
        }
      }
      const runtime = state.runtime.get(message.tabId) || createRuntimeState(message.tabId)
      state.runtime.set(message.tabId, runtime)
      const tab = await safeGetTab(message.tabId)
      await initializeMonitors(message.tabId, config, runtime, tab)
    }
    await persistRules()
    if (config.active) {
      await scheduleRule(message.tabId, 'update')
    }
    return { ok: true }
  },
  async GLOBAL_COMMAND (message) {
    await ensureInitialized()
    switch (message.command) {
      case 'pauseAll':
        await pauseAll()
        break
      case 'resumeAll':
        await resumeAll()
        break
      case 'resetAll':
        await resetAll()
        break
      default:
        break
    }
    return { ok: true, data: state.global }
  },
  async GUARD_UPDATE (message, sender) {
    await ensureInitialized()
    const tabId = message.tabId ?? sender.tab?.id
    const runtime = state.runtime.get(tabId) || createRuntimeState(tabId)
    runtime.guardState = {
      ...runtime.guardState,
      ...message.payload?.guardState
    }
    runtime.offline = Boolean(message.payload?.offline)
    runtime.wasDiscarded = Boolean(message.payload?.wasDiscarded)
    runtime.lastMessageAt = Date.now()
    state.runtime.set(tabId, runtime)
    await persistRuntime(tabId)
    if (state.rules.get(tabId)?.active) {
      await evaluateConditions(tabId, 'guard-update')
    }
    return { ok: true }
  },
  async OPTIONS_GET_SETTINGS () {
    await ensureInitialized()
    return { ok: true, data: state.settings }
  },
  async OPTIONS_SAVE_SETTINGS (message) {
    await ensureInitialized()
    state.settings = {
      ...state.settings,
      ...message.payload,
      guards: {
        ...state.settings.guards,
        ...(message.payload?.guards || {})
      },
      quietHours: {
        ...state.settings.quietHours,
        ...(message.payload?.quietHours || {})
      }
    }
    await chrome.storage.sync.set({ [STORAGE_KEYS.SETTINGS]: state.settings })
    return { ok: true, data: state.settings }
  },
  async OPTIONS_EXPORT () {
    await ensureInitialized()
    const serializableRules = {}
    for (const [tabId, config] of state.rules.entries()) {
      serializableRules[tabId] = config
    }
    const payload = {
      rules: serializableRules,
      settings: state.settings,
      exportedAt: new Date().toISOString()
    }
    return { ok: true, data: payload }
  },
  async OPTIONS_IMPORT (message) {
    await ensureInitialized()
    const { data } = message
    if (!data || typeof data !== 'object') {
      return { ok: false, error: 'Invalid import payload' }
    }
    const newRules = {}
    for (const [tabId, config] of Object.entries(data.rules || {})) {
      newRules[tabId] = normalizeRuleConfig(Number(tabId), config)
    }
    state.rules = new Map(Object.entries(newRules).map(([tabId, config]) => [Number(tabId), config]))
    await persistRules()
    return { ok: true }
  },
  async TEST_SET_MODE (message) {
    state.testMode = Boolean(message.enabled)
    return { ok: true, data: { testMode: state.testMode } }
  }
}

async function initializeMonitors (tabId, config, runtime, tab) {
  if (!config.monitors?.text?.enabled && !config.monitors?.title?.enabled) {
    return
  }

  runtime.changeCount = 0
  let changed = true

  if (config.monitors?.title?.enabled && tab?.title) {
    runtime.titleBaseline = tab.title
    changed = true
  }

  if (config.monitors?.text?.enabled && config.monitors.text.selector) {
    const result = await querySelectorContent(tabId, config.monitors.text.selector)
    runtime.textSnapshot = result?.text || null
    changed = true
  }

  if (changed) {
    await persistRuntime(tabId)
  }
}

async function evaluateMonitors (tabId, config, runtime, tab) {
  const monitors = config.monitors || {}

  if (monitors.title?.enabled) {
    const baseline = runtime.titleBaseline ?? tab?.title
    if (!runtime.titleBaseline && tab?.title) {
      runtime.titleBaseline = tab.title
      await persistRuntime(tabId)
    } else if (monitors.title.stopOnChange && baseline && tab?.title && tab.title !== baseline) {
      appendChangeLog(config, {
        type: 'title-change',
        at: Date.now(),
        details: {
          from: baseline,
          to: tab.title
        }
      })
      runtime.changeCount = (runtime.changeCount || 0) + 1
      await Promise.all([persistRules(), persistRuntime(tabId)])
      return {
        stop: true,
        reason: 'title-monitor',
        badgeText: String(Math.min(runtime.changeCount, 99)),
        notification: {
          title: 'Title changed',
          message: `Tab title changed to “${tab.title}”. Timer stopped.`
        }
      }
    }
  }

  if (monitors.text?.enabled && monitors.text.selector) {
    const result = await querySelectorContent(tabId, monitors.text.selector)
    if (result?.found) {
      const normalizedText = (result.text || '').trim()
      const baseline = runtime.textSnapshot
      const matchTerm = monitors.text.match?.trim()
      const shouldStopForMatch = matchTerm
        ? normalizedText.includes(matchTerm)
        : (baseline && baseline !== normalizedText)

      if (!baseline) {
        runtime.textSnapshot = normalizedText
        await persistRuntime(tabId)
      } else if (shouldStopForMatch) {
        appendChangeLog(config, {
          type: 'text-change',
          at: Date.now(),
          details: {
            selector: monitors.text.selector,
            match: matchTerm,
            previous: baseline.slice(0, 200),
            current: normalizedText.slice(0, 200)
          }
        })
        runtime.changeCount = (runtime.changeCount || 0) + 1
        await Promise.all([persistRules(), persistRuntime(tabId)])
        return {
          stop: true,
          reason: 'text-monitor',
          badgeText: String(Math.min(runtime.changeCount, 99)),
          notification: {
            title: 'Text change detected',
            message: matchTerm
              ? `Found “${matchTerm}” in ${monitors.text.selector}. Timer stopped.`
              : `Text changed for ${monitors.text.selector}. Timer stopped.`
          }
        }
      }
    }
  }

  return { stop: false }
}

async function querySelectorContent (tabId, selector) {
  try {
    const [result] = await chrome.scripting.executeScript({
      target: { tabId },
      func: sel => {
        const element = document.querySelector(sel)
        if (!element) return { found: false, text: null }
        const text = element.innerText || element.textContent || ''
        return { found: true, text }
      },
      args: [selector]
    })
    return result?.result
  } catch (error) {
    console.warn('Ultra Reload query selector failed', selector, error)
    return null
  }
}

function appendChangeLog (config, entry) {
  if (!Array.isArray(config.changeLog)) {
    config.changeLog = []
  }
  config.changeLog.push(entry)
  if (config.changeLog.length > CHANGE_LOG_LIMIT) {
    config.changeLog.splice(0, config.changeLog.length - CHANGE_LOG_LIMIT)
  }
}

async function sendNotification (title, message) {
  const hasPermission = chrome.runtime.getManifest().permissions?.includes('notifications')
  if (!hasPermission) return
  try {
    await chrome.notifications.create({
      type: 'basic',
      iconUrl: chrome.runtime.getURL('img/icons/128.png'),
      title,
      message
    })
  } catch (error) {
    console.warn('Ultra Reload notification failed', error)
  }
}

chrome.runtime.onStartup?.addListener(() => {
  bootstrap().catch(err => console.error('Ultra Reload bootstrap on startup failed', err))
})
