const i18n = key => chrome.i18n.getMessage(key) || key

const elements = {
  statusLabel: document.getElementById('status-label'),
  statusPill: document.getElementById('status-pill'),
  countdown: document.getElementById('countdown-value'),
  guardFlags: document.getElementById('guard-flags'),
  offlineBanner: document.getElementById('offline-banner'),
  discardBanner: document.getElementById('discard-banner'),
  startBtn: document.getElementById('start-btn'),
  stopBtn: document.getElementById('stop-btn'),
  pauseBtn: document.getElementById('pause-btn'),
  resumeBtn: document.getElementById('resume-btn'),
  presetGrid: document.getElementById('preset-grid'),
  intervalInput: document.getElementById('interval-input'),
  jitterInput: document.getElementById('jitter-input'),
  jitterValue: document.getElementById('jitter-value'),
  stopAfterInput: document.getElementById('stop-after-input'),
  hardReload: document.getElementById('hard-reload-toggle'),
  keepActive: document.getElementById('keep-active-toggle'),
  guardTyping: document.getElementById('guard-typing'),
  guardMedia: document.getElementById('guard-media'),
  guardUpload: document.getElementById('guard-upload'),
  textMonitorToggle: document.getElementById('text-monitor-toggle'),
  monitorSelector: document.getElementById('monitor-selector'),
  monitorMatch: document.getElementById('monitor-match'),
  monitorPickerBtn: document.getElementById('monitor-picker-btn'),
  titleMonitorToggle: document.getElementById('title-monitor-toggle'),
  monitorNote: document.querySelector('.monitor-note'),
  globalPauseBtn: document.getElementById('global-pause-btn'),
  globalResumeBtn: document.getElementById('global-resume-btn'),
  globalResetBtn: document.getElementById('global-reset-btn'),
  optionsLink: document.getElementById('options-link')
}

let currentTabId = null
let countdownTimer = null
let viewState = {
  config: null,
  runtime: null,
  settings: null,
  global: null
}

applyI18n()
init().catch(err => {
  console.error('Ultra Reload popup init failed', err)
  elements.statusLabel.textContent = err.message || 'Failed to load extension state'
})

async function init () {
  const [tab] = await chrome.tabs.query({ active: true, currentWindow: true })
  if (!tab) {
    throw new Error('No active tab found.')
  }
  currentTabId = tab.id

  bindEvents()
  await refreshState()
}

function applyI18n () {
  document.querySelectorAll('[data-i18n]').forEach(node => {
    const key = node.getAttribute('data-i18n')
    const message = i18n(key)
    if (message) {
      if (node.tagName === 'INPUT' && node.hasAttribute('placeholder')) {
        node.setAttribute('placeholder', message)
      } else {
        node.textContent = message
      }
    }
  })
}

function bindEvents () {
  elements.presetGrid.addEventListener('click', event => {
    const button = event.target.closest('button[data-interval]')
    if (!button) return
    const intervalMs = Number(button.dataset.interval)
    if (!intervalMs) return
    elements.intervalInput.value = Math.round(intervalMs / 1000)
    setActivePreset(intervalMs)
  })

  elements.jitterInput.addEventListener('input', () => {
    elements.jitterValue.textContent = `${elements.jitterInput.value}%`
  })

  elements.startBtn.addEventListener('click', async () => {
    await withUiLock(async () => {
      await ensureHostPermission(currentTabId)
      const options = collectOptionsFromUi()
      const response = await sendMessage('START_RULE', { tabId: currentTabId, options })
      if (!response.ok) throw new Error(response.error)
      await refreshState()
    })
  })

  elements.stopBtn.addEventListener('click', async () => {
    await withUiLock(async () => {
      const response = await sendMessage('STOP_RULE', { tabId: currentTabId, reason: 'popup' })
      if (!response.ok) throw new Error(response.error)
      await refreshState()
    })
  })

  elements.pauseBtn.addEventListener('click', async () => {
    await withUiLock(async () => {
      const response = await sendMessage('PAUSE_RULE', { tabId: currentTabId })
      if (!response.ok) throw new Error(response.error)
      await refreshState()
    })
  })

  elements.resumeBtn.addEventListener('click', async () => {
    await withUiLock(async () => {
      const response = await sendMessage('RESUME_RULE', { tabId: currentTabId })
      if (!response.ok) throw new Error(response.error)
      await refreshState()
    })
  })

  elements.hardReload.addEventListener('change', handleConfigPatch)
  elements.keepActive.addEventListener('change', handleConfigPatch)
  elements.guardTyping.addEventListener('change', handleConfigPatch)
  elements.guardMedia.addEventListener('change', handleConfigPatch)
  elements.guardUpload.addEventListener('change', handleConfigPatch)
  elements.jitterInput.addEventListener('change', handleConfigPatch)
  elements.stopAfterInput.addEventListener('change', handleConfigPatch)
  elements.intervalInput.addEventListener('change', handleConfigPatch)
  elements.textMonitorToggle.addEventListener('change', handleConfigPatch)
  elements.monitorSelector.addEventListener('change', handleConfigPatch)
  elements.monitorMatch.addEventListener('change', handleConfigPatch)
  elements.titleMonitorToggle.addEventListener('change', handleConfigPatch)

  elements.monitorPickerBtn.addEventListener('click', async () => {
    await withUiLock(async () => {
      const selector = await pickElementSelector()
      if (selector) {
        elements.monitorSelector.value = selector
        elements.textMonitorToggle.checked = true
        await handleConfigPatch()
      }
    })
  })

  elements.globalPauseBtn.addEventListener('click', async () => {
    await withUiLock(async () => {
      await sendMessage('GLOBAL_COMMAND', { command: 'pauseAll' })
      await refreshState()
    })
  })

  elements.globalResumeBtn.addEventListener('click', async () => {
    await withUiLock(async () => {
      await sendMessage('GLOBAL_COMMAND', { command: 'resumeAll' })
      await refreshState()
    })
  })

  elements.globalResetBtn.addEventListener('click', async () => {
    await withUiLock(async () => {
      await sendMessage('GLOBAL_COMMAND', { command: 'resetAll' })
      await refreshState()
    })
  })

  elements.optionsLink.addEventListener('click', () => {
    chrome.runtime.openOptionsPage()
  })
}

function collectOptionsFromUi () {
  const intervalSec = Number(elements.intervalInput.value)
  const intervalMs = Math.max(5, intervalSec) * 1000
  const jitterPercent = Number(elements.jitterInput.value)
  const stopAfterValue = elements.stopAfterInput.value ? Number(elements.stopAfterInput.value) : null
  const options = {
    intervalMs,
    jitter: jitterPercent / 100,
    hardReload: elements.hardReload.checked,
    keepActive: elements.keepActive.checked,
    stopAfter: stopAfterValue,
    guards: {
      pauseOnTyping: elements.guardTyping.checked,
      pauseOnMedia: elements.guardMedia.checked,
      pauseOnUpload: elements.guardUpload.checked
    },
    monitors: {
      text: {
        enabled: elements.textMonitorToggle.checked && !!elements.monitorSelector.value,
        selector: elements.monitorSelector.value,
        match: elements.monitorMatch.value,
        stopOnMatch: true
      },
      title: {
        enabled: elements.titleMonitorToggle.checked,
        stopOnChange: true
      }
    }
  }
  return options
}

function isSupportedUrl (url) {
  try {
    const parsed = new URL(url)
    return parsed.protocol === 'http:' || parsed.protocol === 'https:'
  } catch {
    return false
  }
}

async function ensureHostPermission (tabId) {
  const tab = await chrome.tabs.get(tabId)
  if (!tab?.url || !isSupportedUrl(tab.url)) {
    throw new Error('Switch to an HTTP or HTTPS page to start Auto Reload.')
  }
  const origin = new URL(tab.url).origin
  const pattern = `${origin}/*`
  const hasAccess = await chrome.permissions.contains({ origins: [pattern] })
  if (hasAccess) return true
  const granted = await chrome.permissions.request({ origins: [pattern] })
  if (!granted) {
    throw new Error('Ultra Reload needs site access. Try again after granting permission.')
  }
  return true
}

async function handleConfigPatch () {
  if (!viewState.config?.active) return
  const options = collectOptionsFromUi()
  await withUiLock(async () => {
    const response = await sendMessage('UPDATE_RULE', { tabId: currentTabId, options })
    if (!response.ok) throw new Error(response.error)
    await refreshState()
  })
}

async function refreshState () {
  const response = await sendMessage('GET_STATE', { tabId: currentTabId })
  if (!response.ok) throw new Error(response.error)
  viewState = response.data
  renderState()
}

function renderState () {
  clearInterval(countdownTimer)
  countdownTimer = null

  const { config, runtime, global, settings } = viewState
  const isActive = Boolean(config?.active)
  const manualPause = Boolean(runtime?.manualPause)
  const guardPaused = runtime?.status === 'guard-paused'
  const offline = runtime?.offline
  const discarded = runtime?.wasDiscarded
  const globalPaused = Boolean(global?.paused)

  elements.stopBtn.disabled = !isActive
  elements.pauseBtn.classList.toggle('hidden', !isActive || manualPause)
  elements.resumeBtn.classList.toggle('hidden', !isActive || !manualPause)
  elements.startBtn.disabled = isActive
  elements.pauseBtn.disabled = !isActive
  elements.resumeBtn.disabled = !manualPause

  elements.hardReload.checked = config?.hardReload ?? false
  elements.keepActive.checked = config?.keepActive ?? settings?.keepActiveDefault ?? false

  const jitterPercent = Math.round((config?.jitter ?? settings?.defaultJitter ?? 0.03) * 100)
  elements.jitterInput.value = String(Math.min(5, jitterPercent))
  elements.jitterValue.textContent = `${elements.jitterInput.value}%`

  const intervalSec = Math.round((config?.intervalMs ?? settings?.defaultIntervalMs ?? 60000) / 1000)
  elements.intervalInput.value = intervalSec
  setActivePreset((config?.intervalMs ?? 0))

  if (config?.stopAfter) {
    elements.stopAfterInput.value = config.stopAfter
  } else {
    elements.stopAfterInput.value = ''
  }

  elements.guardTyping.checked = config?.guards?.pauseOnTyping ?? settings?.guards?.pauseOnTyping ?? true
  elements.guardMedia.checked = config?.guards?.pauseOnMedia ?? settings?.guards?.pauseOnMedia ?? true
  elements.guardUpload.checked = config?.guards?.pauseOnUpload ?? settings?.guards?.pauseOnUpload ?? true

  const guardReasons = []
  if (runtime?.guardState?.typing) guardReasons.push(i18n('statusGuardTyping'))
  if (runtime?.guardState?.media || runtime?.guardState?.audible) guardReasons.push(i18n('statusGuardMedia'))
  if (runtime?.guardState?.upload) guardReasons.push(i18n('statusGuardUpload'))
  if (runtime?.offline) guardReasons.push(i18n('statusGuardOffline'))
  if (runtime?.wasDiscarded) guardReasons.push(i18n('statusGuardDiscarded'))
  if (runtime?.guardState?.hidden) guardReasons.push(i18n('statusGuardHidden'))

  renderGuardFlags(guardReasons)
  elements.offlineBanner.classList.toggle('hidden', !offline)
  elements.discardBanner.classList.toggle('hidden', !discarded)

  const monitors = config?.monitors || {}
  elements.textMonitorToggle.checked = Boolean(monitors.text?.enabled)
  elements.monitorSelector.value = monitors.text?.selector || ''
  elements.monitorMatch.value = monitors.text?.match || ''
  elements.titleMonitorToggle.checked = Boolean(monitors.title?.enabled)

  const statusKey = !isActive
    ? 'popupStatusStopped'
    : globalPaused
      ? 'popupStatusPaused'
      : manualPause
        ? 'popupStatusPaused'
        : guardPaused || guardReasons.length > 0
          ? 'popupStatusGuarded'
          : 'popupStatusActive'
  elements.statusLabel.textContent = i18n(statusKey)

  elements.statusPill.textContent = isActive ? (intervalSec >= 60 ? `${Math.round(intervalSec / 60)}m` : `${intervalSec}s`) : '—'
  elements.statusPill.style.background = isActive ? '#22c55e' : '#94a3b8'

  if (isActive && runtime?.nextRunAt) {
    updateCountdown(runtime.nextRunAt)
    countdownTimer = setInterval(() => updateCountdown(runtime.nextRunAt), 1000)
  } else {
    elements.countdown.textContent = '--'
  }
}

function renderGuardFlags (flags) {
  elements.guardFlags.innerHTML = ''
  flags.forEach(text => {
    const li = document.createElement('li')
    li.textContent = text
    elements.guardFlags.appendChild(li)
  })
}

function updateCountdown (nextRunAt) {
  const remaining = Math.max(0, nextRunAt - Date.now())
  const seconds = Math.round(remaining / 1000)
  if (seconds >= 60) {
    const minutes = Math.floor(seconds / 60)
    const secPart = seconds % 60
    elements.countdown.textContent = `${minutes}m ${secPart}s`
  } else {
    elements.countdown.textContent = `${seconds}s`
  }
}

function setActivePreset (intervalMs) {
  document.querySelectorAll('.preset-button').forEach(button => {
    const matches = Number(button.dataset.interval) === intervalMs
    button.classList.toggle('active', matches)
  })
}

async function pickElementSelector () {
  try {
    const [{ result }] = await chrome.scripting.executeScript({
      target: { tabId: currentTabId },
      func: () => {
        return new Promise(resolve => {
          const overlay = document.createElement('div')
          overlay.style.position = 'fixed'
          overlay.style.top = '0'
          overlay.style.left = '0'
          overlay.style.right = '0'
          overlay.style.bottom = '0'
          overlay.style.zIndex = '2147483647'
          overlay.style.pointerEvents = 'none'
          overlay.style.border = '2px solid transparent'
          document.documentElement.appendChild(overlay)

          let highlight = null
          let previousOutline = null

          const cleanup = selector => {
            document.removeEventListener('mousemove', onMove, true)
            document.removeEventListener('click', onClick, true)
            document.removeEventListener('keydown', onKeydown, true)
            if (highlight) {
              highlight.style.outline = previousOutline
              highlight = null
            }
            overlay.remove()
            resolve({ selector })
          }

          const onMove = event => {
            const target = event.target
            if (!(target instanceof HTMLElement)) return
            if (highlight === target) return
            if (highlight) {
              highlight.style.outline = previousOutline
            }
            highlight = target
            previousOutline = target.style.outline
            target.style.outline = '2px solid #38bdf8'
          }

          const cssPath = element => {
            if (!(element instanceof Element)) return ''
            const path = []
            let current = element
            while (current && path.length < 5) {
              let selector = current.nodeName.toLowerCase()
              if (current.id) {
                selector += `#${current.id}`
                path.unshift(selector)
                break
              } else {
                const classList = Array.from(current.classList).filter(Boolean)
                if (classList.length) {
                  selector += '.' + classList.slice(0, 2).join('.')
                }
                path.unshift(selector)
                const parent = current.parentElement
                if (parent) {
                  const siblings = Array.from(parent.children).filter(child => child.nodeName === current.nodeName)
                  if (siblings.length > 1) {
                    const index = siblings.indexOf(current) + 1
                    selector += `:nth-of-type(${index})`
                    path[0] = selector
                  }
                }
                current = parent
              }
            }
            return path.join(' > ')
          }

          const onClick = event => {
            event.preventDefault()
            event.stopPropagation()
            cleanup(cssPath(event.target))
          }

          const onKeydown = event => {
            if (event.key === 'Escape') {
              event.preventDefault()
              cleanup('')
            }
          }

          document.addEventListener('mousemove', onMove, true)
          document.addEventListener('click', onClick, true)
          document.addEventListener('keydown', onKeydown, true)
        })
      }
    })
    return result?.selector || ''
  } catch (error) {
    console.warn('Element picker failed', error)
    return ''
  }
}

async function sendMessage (type, payload = {}) {
  try {
    const response = await chrome.runtime.sendMessage({ type, ...payload })
    return response || { ok: false, error: 'No response' }
  } catch (error) {
    return { ok: false, error: error.message }
  }
}

async function withUiLock (fn) {
  setUiDisabled(true)
  try {
    await fn()
  } catch (error) {
    console.error(error)
    elements.statusLabel.textContent = error.message || String(error)
  } finally {
    setUiDisabled(false)
  }
}

function setUiDisabled (disabled) {
  document.querySelectorAll('button, input').forEach(node => {
    node.disabled = disabled && node.id !== 'options-link'
  })
}
