const DEFAULTS = Object.freeze({
  defaultIntervalMs: 60000,
  defaultJitter: 0.03,
  defaultStopAfter: null,
  guards: {
    pauseOnTyping: true,
    pauseOnMedia: true,
    pauseOnUpload: true
  },
  quietHours: {
    enabled: false,
    start: '22:00',
    end: '07:00'
  },
  keepActiveDefault: false
})

const els = {
  defaultInterval: document.getElementById('default-interval'),
  defaultJitter: document.getElementById('default-jitter'),
  defaultStopAfter: document.getElementById('default-stop-after'),
  guardTyping: document.getElementById('default-guard-typing'),
  guardMedia: document.getElementById('default-guard-media'),
  guardUpload: document.getElementById('default-guard-upload'),
  keepActive: document.getElementById('default-keep-active'),
  saveDefaults: document.getElementById('save-defaults'),
  resetDefaults: document.getElementById('reset-defaults'),
  quietEnabled: document.getElementById('quiet-enabled'),
  quietStart: document.getElementById('quiet-start'),
  quietEnd: document.getElementById('quiet-end'),
  exportBtn: document.getElementById('export-btn'),
  copyExport: document.getElementById('copy-export'),
  downloadExport: document.getElementById('download-export'),
  exportOutput: document.getElementById('export-output'),
  importInput: document.getElementById('import-input'),
  importBtn: document.getElementById('import-btn'),
  clearImport: document.getElementById('clear-import'),
  openDemo: document.getElementById('open-demo')
}

let currentSettings = structuredClone(DEFAULTS)

if (document.readyState === 'loading') {
  document.addEventListener('DOMContentLoaded', start)
} else {
  start()
}

function start () {
  bootstrap().catch(error => {
    console.error('Ultra Reload options init failed', error)
    alert(`Failed to load settings: ${error.message}`)
  })
}

async function bootstrap () {
  await loadSettings()
  bindEvents()
}

function bindEvents () {
  els.saveDefaults.addEventListener('click', async () => {
    await persistSettings()
  })

  els.resetDefaults.addEventListener('click', async () => {
    populateForm(DEFAULTS)
    await persistSettings()
  })

  els.quietEnabled.addEventListener('change', () => {
    setQuietInputsEnabled(els.quietEnabled.checked)
  })

  els.exportBtn.addEventListener('click', async () => {
    const payload = await sendMessage('OPTIONS_EXPORT')
    if (!payload.ok) {
      alert(payload.error || 'Export failed')
      return
    }
    const json = JSON.stringify(payload.data, null, 2)
    els.exportOutput.value = json
  })

  els.copyExport.addEventListener('click', async () => {
    if (!els.exportOutput.value) {
      alert('Nothing to copy. Click Export first.')
      return
    }
    try {
      await navigator.clipboard.writeText(els.exportOutput.value)
      alert('Export JSON copied to clipboard.')
    } catch {
      alert('Clipboard permission denied. Copy manually.')
    }
  })

  els.downloadExport.addEventListener('click', () => {
    if (!els.exportOutput.value) {
      alert('Nothing to download. Click Export first.')
      return
    }
    const blob = new Blob([els.exportOutput.value], { type: 'application/json' })
    const blobUrl = URL.createObjectURL(blob)
    const filename = `ultra-reload-export-${Date.now()}.json`
    const anchor = document.createElement('a')
    anchor.href = blobUrl
    anchor.download = filename
    document.body.appendChild(anchor)
    anchor.click()
    anchor.remove()
    URL.revokeObjectURL(blobUrl)
  })

  els.importBtn.addEventListener('click', async () => {
    const text = els.importInput.value.trim()
    if (!text) {
      alert('Paste JSON before importing.')
      return
    }
    try {
      const parsed = JSON.parse(text)
      const response = await sendMessage('OPTIONS_IMPORT', { data: parsed })
      if (!response.ok) {
        alert(response.error || 'Import failed.')
        return
      }
      alert('Rules imported successfully.')
    } catch (error) {
      alert(`Invalid JSON: ${error.message}`)
    }
  })

  els.clearImport.addEventListener('click', () => {
    els.importInput.value = ''
  })

  els.openDemo.addEventListener('click', () => {
    const url = chrome.runtime.getURL('demo/demo.html')
    chrome.tabs.create({ url })
  })
}

async function loadSettings () {
  const response = await sendMessage('OPTIONS_GET_SETTINGS')
  if (!response.ok) {
    throw new Error(response.error || 'Unable to fetch settings')
  }
  currentSettings = {
    ...DEFAULTS,
    ...(response.data || {}),
    guards: {
      ...DEFAULTS.guards,
      ...(response.data?.guards || {})
    },
    quietHours: {
      ...DEFAULTS.quietHours,
      ...(response.data?.quietHours || {})
    }
  }
  populateForm(currentSettings)
}

function populateForm (settings) {
  els.defaultInterval.value = Math.round(settings.defaultIntervalMs / 1000)
  els.defaultJitter.value = Math.round((settings.defaultJitter ?? 0.03) * 100)
  els.defaultStopAfter.value = settings.defaultStopAfter ?? ''
  els.guardTyping.checked = settings.guards.pauseOnTyping
  els.guardMedia.checked = settings.guards.pauseOnMedia
  els.guardUpload.checked = settings.guards.pauseOnUpload
  els.keepActive.checked = Boolean(settings.keepActiveDefault)
  els.quietEnabled.checked = Boolean(settings.quietHours.enabled)
  els.quietStart.value = settings.quietHours.start
  els.quietEnd.value = settings.quietHours.end
  setQuietInputsEnabled(els.quietEnabled.checked)
}

async function persistSettings () {
  const payload = collectSettingsFromForm()
  const response = await sendMessage('OPTIONS_SAVE_SETTINGS', { payload })
  if (!response.ok) {
    alert(response.error || 'Failed to save settings.')
    return
  }
  currentSettings = response.data
  alert('Settings saved.')
}

function collectSettingsFromForm () {
  const intervalMs = Math.max(5, Number(els.defaultInterval.value)) * 1000
  const jitter = Math.min(5, Math.max(0, Number(els.defaultJitter.value))) / 100
  const stopAfter = els.defaultStopAfter.value ? Number(els.defaultStopAfter.value) : null
  return {
    defaultIntervalMs: intervalMs,
    defaultJitter: jitter,
    defaultStopAfter: stopAfter,
    guards: {
      pauseOnTyping: els.guardTyping.checked,
      pauseOnMedia: els.guardMedia.checked,
      pauseOnUpload: els.guardUpload.checked
    },
    keepActiveDefault: els.keepActive.checked,
    quietHours: {
      enabled: els.quietEnabled.checked,
      start: els.quietStart.value || DEFAULTS.quietHours.start,
      end: els.quietEnd.value || DEFAULTS.quietHours.end
    }
  }
}

function setQuietInputsEnabled (enabled) {
  els.quietStart.disabled = !enabled
  els.quietEnd.disabled = !enabled
  els.quietStart.parentElement.classList.toggle('disabled', !enabled)
  els.quietEnd.parentElement.classList.toggle('disabled', !enabled)
}

async function sendMessage (type, payload = {}) {
  try {
    return await chrome.runtime.sendMessage({ type, ...payload })
  } catch (error) {
    return { ok: false, error: error.message }
  }
}
