// Content guard for Ultra Reload — monitors user activity to pause refresh safely.

const state = {
  typing: false,
  media: false,
  upload: false,
  hidden: document.hidden,
  offline: !navigator.onLine,
  wasDiscarded: Boolean(document.wasDiscarded)
}

let typingTimer = null
let activityTimer = null
let pendingMessage = false

const THROTTLE_MS = 200
const ACTIVITY_PAUSE_MS = 2000

function scheduleSend () {
  if (pendingMessage) return
  pendingMessage = true
  setTimeout(() => {
    pendingMessage = false
    sendUpdate()
  }, THROTTLE_MS)
}

function setState (key, value) {
  if (state[key] === value) return
  state[key] = value
  scheduleSend()
}

function noteTyping () {
  setState('typing', true)
  clearTimeout(typingTimer)
  typingTimer = setTimeout(() => {
    setState('typing', false)
  }, ACTIVITY_PAUSE_MS)
}

function noteActivityPause () {
  setState('typing', true)
  clearTimeout(activityTimer)
  activityTimer = setTimeout(() => {
    setState('typing', false)
  }, ACTIVITY_PAUSE_MS)
}

function refreshMediaState () {
  const mediaElements = Array.from(document.querySelectorAll('audio,video'))
  const hasActive = mediaElements.some(el => {
    try {
      return !el.paused && !el.muted && el.currentTime > 0 && !el.ended && el.volume > 0
    } catch {
      return false
    }
  })
  setState('media', hasActive)
}

function refreshUploadState () {
  const uploadInputs = Array.from(document.querySelectorAll('input[type="file"]'))
  const hasFiles = uploadInputs.some(input => input.files && input.files.length > 0)
  setState('upload', hasFiles)
}

function sendUpdate () {
  chrome.runtime.sendMessage({
    type: 'GUARD_UPDATE',
    payload: {
      guardState: {
        typing: state.typing,
        media: state.media,
        upload: state.upload,
        hidden: state.hidden
      },
      offline: state.offline,
      wasDiscarded: state.wasDiscarded
    }
  }).catch(() => {
    // ignored
  })
}

// Event listeners for typing / activity.
window.addEventListener('keydown', noteTyping, true)
window.addEventListener('input', noteTyping, true)
window.addEventListener('compositionstart', noteTyping, true)
window.addEventListener('compositionupdate', noteTyping, true)
window.addEventListener('compositionend', noteTyping, true)
window.addEventListener('mousedown', noteActivityPause, true)
window.addEventListener('pointerdown', noteActivityPause, true)
window.addEventListener('touchstart', noteActivityPause, true)

document.addEventListener('visibilitychange', () => {
  setState('hidden', document.hidden)
})

window.addEventListener('online', () => {
  setState('offline', false)
})

window.addEventListener('offline', () => {
  setState('offline', true)
})

// Observe uploads.
document.addEventListener('change', event => {
  if (event.target instanceof HTMLInputElement && event.target.type === 'file') {
    refreshUploadState()
  }
}, true)

document.addEventListener('reset', () => {
  refreshUploadState()
}, true)

// Observe media elements.
function bindMediaListeners (root = document) {
  const mediaElements = Array.from(root.querySelectorAll('audio,video'))
  mediaElements.forEach(el => {
    const sync = () => refreshMediaState()
    el.addEventListener('play', sync, true)
    el.addEventListener('pause', sync, true)
    el.addEventListener('ended', sync, true)
    el.addEventListener('volumechange', sync, true)
  })
}

bindMediaListeners()

const observer = new MutationObserver(mutations => {
  for (const mutation of mutations) {
    mutation.addedNodes.forEach(node => {
      if (node instanceof HTMLElement) {
        if (node.matches && node.matches('audio,video')) {
          bindMediaListeners(node)
        } else {
          bindMediaListeners(node)
        }
        if (node.matches && node.matches('input[type="file"]')) {
          refreshUploadState()
        }
      }
    })
  }
  refreshMediaState()
})

observer.observe(document.documentElement, {
  childList: true,
  subtree: true
})

refreshMediaState()
refreshUploadState()
sendUpdate()
